import unittest
import subprocess
import mock
from ..generic import safe_syscall
from ..ovpn import ovpn_status, ovpn_status_parse, assert_safe_for_bash, \
    ovpn_create_conf, ovpn_list_conf, ovpn_delete_conf, ovpn_kill_conn, \
    ovpn_initialize, ovpn_gettlsauth
from ..fileassets import fileassets
from ..cert import ca_initialize
import pytest


class TestOvpn(unittest.TestCase):

    def test_ovpn_create_conf(self):
        cn = '0101010'
        conf = 'push "route 192.168.42.212"'
        proxy = 'vpnnode1'

        # Rewrite the command to a safe local one
        orig_popen = subprocess.Popen

        def mocked_call(cmd, **kw):
            return orig_popen(
                ['bash', '-c', 'cat > /tmp/ovpn_config.conf'],
                **kw
            )

        with mock.patch('subprocess.Popen', side_effect=mocked_call) as patch:
            ovpn_create_conf(cn=cn, conf=conf, proxy=proxy)
            patch.assert_called_with(
                ['ssh', '-o', 'LogLevel=ERROR', proxy,
                 'cat > /etc/openvpn/client-config/{}'.format(cn)
                 ],
                stdin=subprocess.PIPE,
                universal_newlines=True,
            )
        with open('/tmp/ovpn_config.conf') as fh:
            contents = fh.read()

        self.assertEqual(conf, contents)

    @mock.patch('subprocess.check_output',
                return_value='OpenVPN CLIENT LIST')
    def test_ovpn_status_return(self, mock_call):
        status = ovpn_status()
        mock_call.assert_called_with(
            ['ssh', '-o', 'LogLevel=ERROR', 'vpnnode1',
             '/home/mpaproxy/ovpncommand status 3 tun'],
            universal_newlines=True,
        )
        status_ok = status.strip().startswith('OpenVPN CLIENT LIST')
        self.assertTrue(status_ok, 'Wrong status output. Got: %s' % status)

    def test_ovpn_list_conf(self):
        # list configs
        sshcmd = ['ssh', '-o', 'LogLevel=ERROR']
        proxy = 'vpnnode1'
        with mock.patch('subprocess.check_output') as mocked:
            ovpn_list_conf(proxy=proxy)
            mocked.assert_called_with(
                sshcmd + [proxy, 'ls /etc/openvpn/client-config'],
                universal_newlines=True,
            )

    @mock.patch('subprocess.check_call')
    def test_ovpn_delete_conf(self, mocked):
        # delete a config
        proxy = 'vpnnode1'
        cn = '0101010'

        ovpn_delete_conf(cn=cn, proxy=proxy)

    def test_ovpn_kill_conn(self):
        sshopts = ['-o', 'LogLevel=ERROR']
        proxy = 'vpnnode1'
        mode = 'tun'
        basecmd = '/home/mpaproxy/ovpncommand kill'
        expected = basecmd + ' {} ' + mode + '\n'
        orig_popen = subprocess.Popen

        def mpopen(cmd, **kw):
            """Replace by a safe local command"""
            self.assertEqual(cmd[:-1], ['ssh'] + sshopts + [proxy])
            remote_cmd = cmd[-1].replace(basecmd, 'echo ' + basecmd)
            return orig_popen(['bash', '-c', remote_cmd], **kw)

        with mock.patch('subprocess.Popen', side_effect=mpopen) as mocked:
            # Original version with one cn
            cn = '0101010'
            self.assertEqual(
                ovpn_kill_conn(ovpnconn_id=cn, proxy=proxy, mode=mode),
                expected.format(cn)
            )
            mocked.assert_called()

            # New version with list of cns
            cns = ['010101', '101010', '123']
            self.assertEqual(
                ovpn_kill_conn(ovpnconn_id=cns, proxy=proxy, mode=mode),
                ''.join([expected.format(item) for item in cns])
            )

    def test_bash_exploits(self):
        self.assertTrue(assert_safe_for_bash('my-string'))

        with pytest.raises(AssertionError):
            assert_safe_for_bash('echo;')

    def test_ovpn_status(self):
        # only version 3 implemented, all other raise an error
        self.assertRaises(NotImplementedError, ovpn_status, version=23)

    @mock.patch('subprocess.check_output')
    def test_ovpn_status_unexpected(self, mock_call):
        self.assertRaises(AssertionError, ovpn_status, version=3)

    def test_ovpn_status_parse(self):
        # regular results
        contents = fileassets['tests.ovpn_status_3']
        res = ovpn_status_parse(status=contents)
        expected = {
            'clients': [
                1099925,
                1099981,
                1099995,
                1098934,
                1100012,
                1100015,
                1099991,
                1098935,
                1099924,
                1100004,
                1098933,
                1099982,
                1098938,
                1098937,
                1100013,
            ]
        }
        self.assertTrue(res == expected)

        # unicode
        contents = u'CLIENT_LIST\t123'
        res = ovpn_status_parse(status=contents)
        expected = {'clients': [123]}
        self.assertEqual(res, expected)

        # no results
        contents = fileassets['tests.ovpn_status_3_empty']
        res = ovpn_status_parse(status=contents)
        expected = {'clients': []}
        self.assertEqual(res, expected)

        # -- bogus input --
        # None
        res = ovpn_status_parse(status=None)
        expected = {'clients': []}
        self.assertEqual(res, expected)
        # True
        res = ovpn_status_parse(status=True)
        expected = {'clients': []}
        self.assertEqual(res, expected)
        # False
        res = ovpn_status_parse(status=False)
        expected = {'clients': []}
        self.assertEqual(res, expected)
        # Number
        res = ovpn_status_parse(status=123)
        expected = {'clients': []}
        self.assertEqual(res, expected)
        # Nil-String
        self.assertRaises(
            ValueError,
            ovpn_status_parse, status=''
        )
        # Wrong version
        self.assertRaises(
            NotImplementedError,
            ovpn_status_parse, status='asdf', version=23
        )

    def test_ovpn_cert_gen(self):
        # setup a testing CA
        # ca_pass = 'ldskjf;f93579&/k3n4j'
        ca_subj = "/C=DE/ST=NRW/L=Herford/O=perfact::ema/CN=ema-devel-2016"
        appca_id1 = 1337
        ca_initialize(ca_subj, appca_id=appca_id1,
                      ca_path_fmt="/tmp/home/zope/%d")

        # Build OVPN specific keys, cert ...
        subj = '/C=DE/ST=NRW/L=Herford/O=perfact::ema/CN=test-OVPN-Server'
        ovpn_initialize(subj, expire_days=None, passphrase=None,
                        appca_id=appca_id1,
                        ovpn_path='/tmp/ovpn',
                        rsabits=512,
                        dhbits=512,
                        ca_path_fmt="/tmp/home/zope/%d"
                        )

        res, out = safe_syscall(['ls', '-l', '/tmp/ovpn'], raisemode=True)
        print(out)

        res, out = safe_syscall(['openssl', 'dhparam', '-in',
                                '/tmp/ovpn/dhparam.pem', '-check'
                                 ], raisemode=True)
        print(out)

        fh = open('/tmp/ovpn/tlsauth.key', 'r')
        secret = fh.read()
        fh.close()

        assert 'BEGIN OpenVPN Static key V1' in secret, \
            'Start indicator for OVPN static key missing in tlsauth.key!'

        assert 'END OpenVPN Static key V1' in secret, \
            'Stop indicator for OVPN static key missing in tlsauth.key!'

        key = ovpn_gettlsauth(ovpn_path='/tmp/ovpn')
        assert key.startswith('-----BEGIN OpenVPN Static key V1-----')

        # Clean up
        safe_syscall(['rm', '-rf', '/tmp/ovpn'])
        safe_syscall(['rm', '-rf', '/tmp/home/zope'])
